package easik.sketch.vertex;

import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.tree.DefaultMutableTreeNode;

import easik.sketch.attribute.EntityAttribute;
import easik.sketch.attribute.UniqueKey;
import easik.sketch.datatype.DataType;


/**
 * An entity node represents a table in a database. It has a name,
 * attributes, and unique keys. This class keeps track of all these
 * elements. 
 * 
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @version 2006-06-21 Kevin Green
 */
public class EntityNode extends SketchVertex {
	/**
	 * List of all attributes for this entity
	 */
	private ArrayList<EntityAttribute> _entityAttributes;
	/**
	 * List of all unique keys for this entity
	 */
	private ArrayList<UniqueKey> _uniqueKeys;
	/**
	 * Node representing this entity for the information tree
	 */
	private DefaultMutableTreeNode _entityNode;
	/**
	 * Node representing the key for the information tree
	 */
	private DefaultMutableTreeNode _keyNode;
	
	/**
	 * Creates a new entity node with the name provided.
	 * 
	 * @param nodeName The name of the new node
	 */
	public EntityNode( String nodeName ) {
			super();
			_name = nodeName;
			_marked = false;
			_entityAttributes = new ArrayList<EntityAttribute>();
			_uniqueKeys = new ArrayList<UniqueKey>();
	}
	
	/**
	 * Creates a new enity node with the name provided.
	 * Stores visual representation information.
	 * 
	 * @param nodeName Name of the new node
	 * @param x X Coordinate of the new node
	 * @param y Y Coordinate of the new node
	 */
	public EntityNode( String nodeName, int x, int y ) {
			super();
			_posX = x;
			_posY = y;
			_name = nodeName;
			_marked = false;
			_entityAttributes = new ArrayList<EntityAttribute>();
			_uniqueKeys = new ArrayList<UniqueKey>();
	}
	
	/**
	 * Creates a new EntityAttribute and add its to the list of attributes
	 * 
	 * @param inName Name of attribute
	 * @param inAttributeType SQL Type of the attribute
	 */
	public void addAttribute(String inName, DataType inAttributeType){
		_entityAttributes.add(new EntityAttribute(inName, inAttributeType));
	}
	
	/**
	 * Creates a new EntityAttribute and add its to the list of attributes
	 * 
	 * @param inAtt	The EntityAttribute to add to this EntityNode.
	 */
	public void addAttribute(EntityAttribute inAtt){
		_entityAttributes.add(inAtt);
	}
	
	/**
	 * Returns the list of EntityAttributes
	 * 
	 * @return List of EntityAttributes
	 */
	public ArrayList getAttributes(){
		return _entityAttributes;
	}
	
	/**
	 * Removes an attribute from the list
	 * 
	 * @param inAttribute The attribute to be removed
	 */
	public void removeAttribute(EntityAttribute inAttribute){
		_entityAttributes.remove(_entityAttributes.indexOf(inAttribute));
		
		//Loop through keys to remove references to the attribute
		Iterator it = _uniqueKeys.iterator();
		while(it.hasNext()){
			UniqueKey curKey = (UniqueKey) it.next();
			curKey.getAttributes().remove(inAttribute);
		}
		
		UniqueKey.testUnique(this);
	}
	
	/**
	 * Adds a unique key to the list
	 * 
	 * @param inAtts The attributes used in the key
	 * @param inName The name of the unique key
	 */
	public void addUniqueKey(ArrayList inAtts, String inName){
		_uniqueKeys.add(new UniqueKey(this, inAtts, inName));
	}
	
	/**
	 * Adds a unique key to the list
	 * 
	 * @param inKey The key to be added
	 */
	public void addUniqueKey(UniqueKey inKey){
		_uniqueKeys.add(inKey);
	}
	
	/**
	 * Removes a unique key from the list
	 * 
	 * @param inKey The unique key to be removed
	 */
	public void removeUniqueKey(UniqueKey inKey){
		_uniqueKeys.remove(_uniqueKeys.indexOf(inKey));
	}
	
	/**
	 * Returns an ArrayList of the unique keys
	 * 
	 * @return An ArrayList of the unique keys
	 */
	public ArrayList getUniqueKeys(){
		return _uniqueKeys;
	}
	
	/**
	 * Sets the tree node used to display entity
	 * 
	 * @param inNode The tree node used to display entity
	 */
	public void setNode(DefaultMutableTreeNode inNode){
		_entityNode = inNode;
	}
	
	/**
	 * Returns the tree node used to display entity
	 * 
	 * @return The tree node used to display entity
	 */
	public DefaultMutableTreeNode getNode(){
		return _entityNode;
	}
	
	/**
	 * Sets the tree node used to display unique keys
	 * 
	 * @param inNode The tree node used to display unique keys
	 */
	public void setKeyNode(DefaultMutableTreeNode inNode){
		_keyNode = inNode;
	}
	
	/**
	 * Returns the tree node used to display unique keys
	 * 
	 * @return The tree node used to display unique keys
	 */
	public DefaultMutableTreeNode getKeyNode(){
		return _keyNode;
	}
	
	/**
	 * Test to see if the keys tree node should be visible or not
	 */
	public void testKeysNode(){
		if(_uniqueKeys.size() > 0){
			_entityNode.insert(_keyNode, 0);
		}
		else
			_keyNode.removeFromParent();
	}
}
